<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Models\CallbackRetry;
use App\Models\Client;
use App\Models\GameSession;
use App\Models\Transaction;
use App\Models\User;
use App\Services\HttpService;
use App\Services\Logger;
use PDO;

class CallbackController
{
    public function __construct(
        private PDO $pdo,
        private GameSession $gameSession,
        private Transaction $transaction,
        private User $userModel,
        private Client $clientModel,
        private CallbackRetry $callbackRetry,
        private HttpService $http,
        private Logger $logger,
    ) {}

    public function handle(string $rawInput, string $clientIp): array
    {
        $data = json_decode($rawInput, true);
        $this->logger->log(null, '/api/callback', $data ?: ['raw' => substr($rawInput, 0, 500)], null, $clientIp, null);

        if (!$data || !is_array($data)) {
            return $this->errorResponse(-1, 'Invalid JSON 无效JSON');
        }

        $required = ['game_uid', 'game_round', 'member_account', 'bet_amount', 'win_amount'];
        foreach ($required as $field) {
            if (!array_key_exists($field, $data)) {
                return $this->errorResponse(-1, "Missing required field: $field");
            }
        }

        $gameUid = (string) $data['game_uid'];
        $gameRound = (string) $data['game_round'];
        $memberAccount = (string) $data['member_account'];
        $betAmount = (float) $data['bet_amount'];
        $winAmount = (float) $data['win_amount'];

        $existingTx = $this->transaction->findByGameRound($gameRound);
        if ($existingTx) {
            $credit = max(0, (float) $existingTx['bet_amount'] - (float) $existingTx['win_amount']);
            $response = ['credit_amount' => $credit, 'timestamp' => (int) round(microtime(true) * 1000)];
            $this->logger->log($existingTx['client_id'], '/api/callback', null, $response, $clientIp, 200);
            return $response;
        }

        $session = $this->gameSession->findByGameUid($gameUid);
        if (!$session) {
            return $this->errorResponse(-1, 'Unknown game session');
        }

        $clientId = (int) $session['client_id'];
        $userId = $session['user_id'];
        $client = $this->clientModel->getById($clientId);
        if (!$client) {
            return $this->errorResponse(-1, 'Client not found');
        }

        $commissionPercent = (float) ($client['commission_percent'] ?? 0);
        $commission = ($betAmount * $commissionPercent) / 100;
        $creditAmount = max(0, $betAmount - $winAmount);

        try {
            $this->pdo->beginTransaction();

            $user = $this->userModel->lockForUpdate($clientId, $userId);
            if (!$user) {
                $this->pdo->rollBack();
                return $this->errorResponse(-1, 'User not found');
            }

            $beforeBalance = (float) $user['balance'];
            $newBalance = $beforeBalance - $betAmount + $winAmount;
            if ($newBalance < 0) {
                $this->pdo->rollBack();
                return $this->errorResponse(-1, 'Insufficient balance');
            }

            $this->userModel->updateBalance($clientId, $userId, $newBalance);

            $stmt = $this->pdo->prepare('UPDATE clients SET balance = balance + ? WHERE id = ?');
            $stmt->execute([$commission, $clientId]);

            $this->transaction->create([
                'client_id' => $clientId,
                'user_id' => $userId,
                'game_uid' => $gameUid,
                'game_round' => $gameRound,
                'bet_amount' => $betAmount,
                'win_amount' => $winAmount,
                'before_balance' => $beforeBalance,
                'after_balance' => $newBalance,
                'commission_amount' => $commission,
            ]);

            $this->pdo->commit();
        } catch (\Throwable $e) {
            $this->pdo->rollBack();
            $this->logger->log($clientId, '/api/callback', $data, ['error' => $e->getMessage()], $clientIp, 500);
            return $this->errorResponse(-1, 'Processing failed');
        }

        $callbackUrl = $session['callback_url'] ?: $client['callback_url'];
        $forwardPayload = array_merge($data, [
            'credit_amount' => $creditAmount,
            'timestamp' => (int) round(microtime(true) * 1000),
        ]);

        $forwardSuccess = $this->forwardToClient($callbackUrl, $forwardPayload, $clientId, $clientIp);
        if (!$forwardSuccess && $callbackUrl) {
            $this->callbackRetry->add($gameRound, $clientId, $forwardPayload, $callbackUrl);
        }

        $response = [
            'credit_amount' => $creditAmount,
            'timestamp' => (int) round(microtime(true) * 1000),
        ];
        $this->logger->log($clientId, '/api/callback', null, $response, $clientIp, 200);
        return $response;
    }

    private function forwardToClient(string $url, array $payload, int $clientId, string $ip): bool
    {
        if (empty($url)) return true;
        $result = $this->http->postJson($url, $payload);
        $this->logger->log($clientId, 'client_callback', $payload, $result['decoded'] ?? ['raw' => $result['body'] ?? $result['error'] ?? ''], $ip, $result['code'] ?? 0);
        return $result['success'] ?? false;
    }

    private function errorResponse(int $credit, string $message): array
    {
        return ['credit_amount' => $credit, 'error' => $message];
    }
}
