<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Middleware\CheckWhitelist;
use App\Middleware\EncryptPayload;
use App\Middleware\RateLimiter;
use App\Middleware\SendRequest;
use App\Middleware\ValidateApiKey;
use App\Models\GameSession;
use App\Models\User;
use App\Services\Logger;

class LaunchGameController
{
    public function __construct(
        private ValidateApiKey $validateApiKey,
        private CheckWhitelist $checkWhitelist,
        private RateLimiter $rateLimiter,
        private EncryptPayload $encryptPayload,
        private SendRequest $sendRequest,
        private GameSession $gameSession,
        private User $userModel,
        private Logger $logger,
        private string $mainToken,
        private string $mainSecret,
        private string $proxyBaseUrl,
    ) {}

    public function handle(array $input, string $clientIp, ?string $origin): array
    {
        $apiKey = trim((string) ($input['api_key'] ?? ''));
        $userId = trim((string) ($input['user_id'] ?? ''));
        $balanceVal = (float) ($input['balance'] ?? 0);
        $gameUid = trim((string) ($input['game_uid'] ?? ''));
        $returnUrl = trim((string) ($input['return_url'] ?? $input['return'] ?? ''));
        $callbackUrl = trim((string) ($input['callback_url'] ?? $input['callback'] ?? ''));
        $currencyCode = trim((string) ($input['currency_code'] ?? ''));
        $language = trim((string) ($input['language'] ?? ''));

        $this->logger->log(null, '/api/launch-game', $input, null, $clientIp, null);

        $validation = ($this->validateApiKey)($apiKey);
        if (!$validation['valid']) {
            $err = ['code' => 401, 'msg' => $validation['error']];
            $this->logger->log(null, '/api/launch-game', null, $err, $clientIp, 401);
            return $err;
        }

        $client = $validation['client'];
        if (($client['status'] ?? '') !== 'active') {
            $err = ['code' => 403, 'msg' => 'Client is inactive'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 403);
            return $err;
        }

        if (!$this->checkWhitelist->__invoke($client, $clientIp, $origin)) {
            $err = ['code' => 403, 'msg' => 'IP or domain not allowed'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 403);
            return $err;
        }

        if (!$this->rateLimiter->check($apiKey)) {
            $err = ['code' => 429, 'msg' => 'Too many requests'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 429);
            return $err;
        }

        if (empty($userId) || empty($gameUid)) {
            $err = ['code' => 400, 'msg' => 'user_id and game_uid are required'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 400);
            return $err;
        }

        $user = $this->userModel->findOrCreate($client['id'], $userId, $balanceVal);
        if ($balanceVal > 0) {
            $this->userModel->updateBalance($client['id'], $userId, $balanceVal);
        }

        $returnUrl = $returnUrl ?: $client['return_url'] ?? $this->proxyBaseUrl . '/';
        $callbackUrl = $callbackUrl ?: $client['callback_url'] ?? rtrim($this->proxyBaseUrl, '/') . '/api/callback';

        $ourCallbackUrl = rtrim($this->proxyBaseUrl, '/') . '/api/callback';
        $userBalance = (float) ($user['balance'] ?? $balanceVal);

        $payload = [
            'user_id' => $userId,
            'balance' => $userBalance > 0 ? $userBalance : $balanceVal,
            'game_uid' => $gameUid,
            'token' => $this->mainToken,
            'timestamp' => (int) round(microtime(true) * 1000),
            'return' => $returnUrl,
            'callback' => $ourCallbackUrl,
        ];
        if ($currencyCode !== '') $payload['currency_code'] = $currencyCode;
        if ($language !== '') $payload['language'] = $language;

        try {
            $encrypted = ($this->encryptPayload)($payload, $this->mainSecret);
        } catch (\Throwable $e) {
            $err = ['code' => 500, 'msg' => 'Encryption failed'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 500);
            return $err;
        }

        $response = ($this->sendRequest)($encrypted);

        $sessionId = $response['decoded']['data']['session_id'] ?? null;
        $this->gameSession->create($client['id'], $userId, $gameUid, $sessionId, $callbackUrl, $returnUrl);

        $this->logger->log($client['id'], '/api/launch-game', ['payload' => '***'], $response['decoded'] ?? ['raw' => $response['body'] ?? ''], $clientIp, $response['code'] ?? 200);

        if (!$response['success']) {
            return [
                'code' => $response['code'] ?: 500,
                'msg' => $response['error'] ?? ($response['decoded']['msg'] ?? 'Request failed'),
                'data' => $response['decoded']['data'] ?? null,
            ];
        }

        return $response['decoded'] ?? ['code' => 0, 'msg' => 'OK', 'data' => json_decode($response['body'] ?? '{}', true)];
    }
}
