<?php

declare(strict_types=1);

namespace App\Models;

use PDO;

class CallbackRetry
{
    public function __construct(
        private PDO $pdo
    ) {}

    public function add(string $gameRound, int $clientId, array $payload, string $callbackUrl): void
    {
        $stmt = $this->pdo->prepare('
            INSERT INTO callback_retries (game_round, client_id, payload, callback_url, attempts, next_retry_at)
            VALUES (?, ?, ?, ?, 0, NOW())
        ');
        $stmt->execute([$gameRound, $clientId, json_encode($payload), $callbackUrl]);
    }

    public function getPending(int $limit = 50): array
    {
        $stmt = $this->pdo->prepare('
            SELECT * FROM callback_retries
            WHERE attempts < max_attempts AND (next_retry_at IS NULL OR next_retry_at <= NOW())
            ORDER BY next_retry_at ASC LIMIT ?
        ');
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    }

    public function markAttempted(int $id, string $error): void
    {
        $stmt = $this->pdo->prepare('
            UPDATE callback_retries SET attempts = attempts + 1, last_error = ?, next_retry_at = DATE_ADD(NOW(), INTERVAL POWER(2, attempts) MINUTE)
            WHERE id = ?
        ');
        $stmt->execute([$error, $id]);
    }

    public function delete(int $id): void
    {
        $this->pdo->prepare('DELETE FROM callback_retries WHERE id = ?')->execute([$id]);
    }
}
