<?php

declare(strict_types=1);

namespace App\Models;

use PDO;

class User
{
    public function __construct(
        private PDO $pdo
    ) {}

    public function findOrCreate(int $clientId, string $userId, float $initialBalance = 0): array
    {
        $user = $this->findByClientAndUserId($clientId, $userId);
        if ($user) {
            return $user;
        }
        $stmt = $this->pdo->prepare('INSERT INTO users (client_id, user_id, balance) VALUES (?, ?, ?)');
        $stmt->execute([$clientId, $userId, $initialBalance]);
        return $this->findByClientAndUserId($clientId, $userId);
    }

    public function findByClientAndUserId(int $clientId, string $userId): ?array
    {
        $stmt = $this->pdo->prepare('SELECT * FROM users WHERE client_id = ? AND user_id = ?');
        $stmt->execute([$clientId, $userId]);
        return $stmt->fetch() ?: null;
    }

    public function findById(int $id): ?array
    {
        $stmt = $this->pdo->prepare('SELECT * FROM users WHERE id = ?');
        $stmt->execute([$id]);
        return $stmt->fetch() ?: null;
    }

    /** Lock row for update (use within transaction) */
    public function lockForUpdate(int $clientId, string $userId): ?array
    {
        $stmt = $this->pdo->prepare('SELECT * FROM users WHERE client_id = ? AND user_id = ? FOR UPDATE');
        $stmt->execute([$clientId, $userId]);
        return $stmt->fetch() ?: null;
    }

    public function updateBalance(int $clientId, string $userId, float $newBalance): bool
    {
        $stmt = $this->pdo->prepare('UPDATE users SET balance = ?, updated_at = NOW() WHERE client_id = ? AND user_id = ?');
        return $stmt->execute([$newBalance, $clientId, $userId]);
    }

    public function getByClientId(int $clientId, int $limit = 100, int $offset = 0): array
    {
        $stmt = $this->pdo->prepare('SELECT * FROM users WHERE client_id = ? ORDER BY updated_at DESC LIMIT ? OFFSET ?');
        $stmt->execute([$clientId, $limit, $offset]);
        return $stmt->fetchAll();
    }

    public function countByClientId(int $clientId): int
    {
        $stmt = $this->pdo->prepare('SELECT COUNT(*) FROM users WHERE client_id = ?');
        $stmt->execute([$clientId]);
        return (int) $stmt->fetchColumn();
    }
}
